<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreInvoiceRequest;
use App\Http\Requests\UpdateInvoiceRequest;
use App\Models\Invoice;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    // GET /api/v1/invoices?search=REF123&filter[status]=1&filter[customer_id]=5&overdue=1&from=2025-10-01&to=2025-10-31&sort=-issue_date&per_page=25
    public function index(Request $request)
    {
        $q = Invoice::query()->where('created_by', $request->user()->id);

        if ($s = $request->string('search')->toString()) {
            $q->where('ref_number', 'like', "%{$s}%");
        }

        if ($filters = $request->input('filter', [])) {
            foreach (['status','customer_id','category_id'] as $col) {
                if (array_key_exists($col, $filters) && $filters[$col] !== '') {
                    $q->where($col, $filters[$col]);
                }
            }
        }

        if ($request->boolean('overdue')) {
            $q->whereDate('due_date', '<', now()->toDateString())
              ->whereIn('status', [0,1,2]); // tweak per your status map
        }

        if ($from = $request->date('from')) $q->whereDate('issue_date','>=',$from);
        if ($to   = $request->date('to'))   $q->whereDate('issue_date','<=',$to);

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort,'-') ? 'desc' : 'asc';
            $col = ltrim($sort,'-');
            $q->orderBy($col,$dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int)$request->query('per_page',15),1),100);
        return response()->json($q->paginate($perPage));
    }

    // POST /api/v1/invoices
    public function store(StoreInvoiceRequest $request)
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;

        // defaults
        $data['status'] = $data['status'] ?? 0;
        $data['shipping_display'] = $data['shipping_display'] ?? 1;
        $data['discount_apply']   = $data['discount_apply'] ?? 0;

        $invoice = Invoice::create($data);
        return response()->json($invoice, 201);
    }

    // GET /api/v1/invoices/{invoice}
    public function show(Request $request, Invoice $invoice)
    {
        if ($invoice->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        // Optionally eager-load customer
        $invoice->loadMissing('customer');
        return response()->json($invoice);
    }

    // PUT/PATCH /api/v1/invoices/{invoice}
    public function update(UpdateInvoiceRequest $request, Invoice $invoice)
    {
        if ($invoice->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        $invoice->update($request->validated());
        return response()->json($invoice);
    }

    // DELETE /api/v1/invoices/{invoice}
    public function destroy(Request $request, Invoice $invoice)
    {
        if ($invoice->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        // If you prefer soft-delete, add SoftDeletes to the model and use $invoice->delete()
        $invoice->delete();
        return response()->json(null, 204);
    }
}
