<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Http\Resources\UserResource;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    // GET /api/v1/users?search=keith&filter[is_active]=1&sort=-created_at&per_page=25
    public function index(Request $request)
    {
        $q = User::query();

        if ($s = $request->string('search')->toString()) {
            $q->where(function ($qq) use ($s) {
                $qq->where('name', 'like', "%{$s}%")
                   ->orWhere('email', 'like', "%{$s}%")
                   ->orWhere('type', 'like', "%{$s}%");
            });
        }

        if ($filters = $request->input('filter', [])) {
            foreach (['is_active','active_status','delete_status','plan','type'] as $col) {
                if (array_key_exists($col, $filters) && $filters[$col] !== '') {
                    $q->where($col, $filters[$col]);
                }
            }
        }

        if ($from = $request->date('from')) $q->whereDate('created_at', '>=', $from);
        if ($to   = $request->date('to'))   $q->whereDate('created_at', '<=', $to);

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort, '-') ? 'desc' : 'asc';
            $col = ltrim($sort, '-');
            $q->orderBy($col, $dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int) $request->query('per_page', 15), 1), 100);
        return UserResource::collection($q->paginate($perPage));
    }

    // POST /api/v1/users
    public function store(StoreUserRequest $request)
    {
        $data = $request->validated();

        // If password empty, unset so mutator doesn't null it
        if (array_key_exists('password', $data) && ($data['password'] ?? '') === '') {
            unset($data['password']);
        }

        $user = User::create($data);
        return (new UserResource($user))->response()->setStatusCode(201);
    }

    // GET /api/v1/users/{user}
    public function show(User $user)
    {
        return new UserResource($user);
    }

    // PUT/PATCH /api/v1/users/{user}
    public function update(UpdateUserRequest $request, User $user)
    {
        $data = $request->validated();

        if (array_key_exists('password', $data) && ($data['password'] ?? '') === '') {
            unset($data['password']); // keep existing password if blank provided
        }

        $user->update($data);
        return new UserResource($user);
    }

    // DELETE /api/v1/users/{user}
    // Logical delete: flip flags instead of hard delete
    public function destroy(User $user)
    {
        $user->update(['is_active' => false, 'active_status' => false, 'delete_status' => 0]);
        return response()->json(null, 204);
    }
}
