<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreVenderRequest;
use App\Http\Requests\UpdateVenderRequest;
use App\Models\Vender;
use Illuminate\Http\Request;

class VenderController extends Controller
{
    // GET /api/v1/venders?search=acme&filter[is_active]=1&filter[company_industry]=IT&sort=-created_at&per_page=25
    public function index(Request $request)
    {
        $q = Vender::query()->where('created_by', $request->user()->id);

        if ($s = $request->string('search')->toString()) {
            $q->where(function ($w) use ($s) {
                $w->where('name','like',"%{$s}%")
                  ->orWhere('email','like',"%{$s}%")
                  ->orWhere('company_name','like',"%{$s}%")
                  ->orWhere('company_city','like',"%{$s}%")
                  ->orWhere('company_state','like',"%{$s}%")
                  ->orWhere('company_telephone','like',"%{$s}%");
            });
        }

        if ($filters = $request->input('filter', [])) {
            foreach (['is_active','company_industry','company_entity_type','company_city','company_state'] as $col) {
                if (array_key_exists($col, $filters) && $filters[$col] !== '') {
                    $q->where($col, $filters[$col]);
                }
            }
        }

        if ($from = $request->date('from')) $q->whereDate('created_at','>=',$from);
        if ($to   = $request->date('to'))   $q->whereDate('created_at','<=',$to);

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort,'-') ? 'desc' : 'asc';
            $col = ltrim($sort,'-');
            $q->orderBy($col,$dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int)$request->query('per_page',15),1),100);
        return response()->json($q->paginate($perPage));
    }

    // POST /api/v1/venders
    public function store(StoreVenderRequest $request)
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;
        $data['is_active']  = $data['is_active'] ?? 1;

        // if password empty, don't touch (mutator only hashes non-empty)
        if (isset($data['password']) && $data['password'] === '') unset($data['password']);

        $vender = Vender::create($data);
        return response()->json($vender, 201);
    }

    // GET /api/v1/venders/{vender}
    public function show(Request $request, Vender $vender)
    {
        if ($vender->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        return response()->json($vender);
    }

    // PUT/PATCH /api/v1/venders/{vender}
    public function update(UpdateVenderRequest $request, Vender $vender)
    {
        if ($vender->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        $data = $request->validated();
        if (array_key_exists('password',$data) && ($data['password'] ?? '') === '') {
            unset($data['password']); // keep existing hash
        }

        $vender->update($data);
        return response()->json($vender);
    }

    // DELETE /api/v1/venders/{vender}
    // We’ll do a logical delete by setting is_active=0 (safer than hard delete).
    public function destroy(Request $request, Vender $vender)
    {
        if ($vender->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        $vender->update(['is_active' => 0]);
        return response()->json(null, 204);
    }
}
